/*

 StringManipLib
 Copyright (c) 2004 PXI Project Team

 Permission is hereby granted, free of charge, to any person obtaining a copy of
 this software and associated documentation files (the "Software"), to deal in the
 Software without restriction, including without limitation the rights to use,
 copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the
 Software, and to permit persons to whom the Software is furnished to do so,
 subject to the following conditions:

 The above copyright notice and this permission notice shall be included in all copies
 or substantial portions of the Software.

 THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED,
 INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
 HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
 OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
 SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

*/

#include <string>
#include <cstring>
#include "stringmaniplib.h"
using namespace std;

/*
 
 format				| Not implemented / Unnecessary
 lowercase			| Implemented
 replace			| Implemented
 replaceall			| Implemented
 str				| Implemented
 strcomp			| Implemented
 titlecase			| Implemented
 uppercase			| Implemented
 val				| Implemented
 capalllower		| Implemented
 capallupper		| Implemented
 capalternating		| Implemented
 capinvert			| Implemented
 caprandom			| Not implemented / Unnecessary
 capsentence		| Implemented
 captitle			| Implemented
 countstuff			| Implemented
 thatmanystrings	| Implemented
 
*/

int asc(const string & s)
{
	return s[0];
}

string chr(int v)
{
	string s = " ";
	s[0] = v;
	return s;
}

int wasc(const string & s)
{
	return s[1] + (256 * s[0]);
}

string wrd(int v)
{
	string s = "  ";
	s[0] = (v / 256);
	s[1] = (v % 256);
	return s;
}

int lasc(const string & s)
{
	return s[3] + (256 * s[2]) + (65536 * s[1]) + (16777216 * s[0]);
}

string lng(int v)
{
	string s = "    ";
	s[0] = ((v / 16777216) % 256);
	s[1] = ((v / 65536) % 256);
	s[2] = ((v / 256) % 256);
	s[3] = (v % 256);
	return s;
}

int instr(const string & s, const string & ss)
{
	int i;
	int sz = (int)ss.size();
	for (i = 0; i<(int)s.size(); ++i)
	{
		if ( strcmpi( s.substr( i, sz ).c_str(), ss.c_str() ) == 0 ) { return i+1; }
	}
	return -1;
}

int instrb(const string & s, const string & ss)
{
	int i;
	int sz = (int)ss.size();
	for (i = 0; i<(int)s.size(); ++i)
	{
		if ( strcmp( s.substr( i, sz ).c_str(), ss.c_str() ) == 0 ) { return i+1; }
	}
	return -1;
}

int rinstr(const string & s, const string & ss)
{
	int i;
	int sz = (int)ss.size();
	for (i = (int)s.size()-1; i>=0; --i)
	{
		if ( strcmpi( s.substr( i, sz ).c_str(), ss.c_str() ) == 0 ) { return i+1; }
	}
	return -1;
}

int rinstrb(const string & s, const string & ss)
{
	int i;
	int sz = (int)ss.size();
	for (i = (int)s.size()-1; i>=0; --i)
	{
		if ( strcmpi( s.substr( i, sz ).c_str(), ss.c_str() ) == 0 ) { return i+1; }
	}
	return -1;
}

string left(const string & s, int cnt)
{
	if (cnt < 1)
	{
		return "";
	}
	if (cnt > (int)s.size()) { return s; }
	return s.substr( 0, cnt );
}

string mid(const string & s, int pos)
{
	if (pos > (int)s.size())
	{
		return "";
	}
	if (pos < 1) { return s; }
	return s.substr( pos-1, (int)s.size() );
}

string mid(const string & s, int pos, int sl)
{
	if (pos > (int)s.size())
	{
		return "";
	}
	if (pos < 1) { return left(s, sl); }
	return s.substr( pos-1, sl );
}


string right(const string & s, int cnt)
{
	if (cnt < 1)
	{
		return "";
	}
	if (cnt > (int)s.size()) { return s; }
	return s.substr( (int)s.size()-cnt, (int)s.size() );
}

int len(const string & s)
{
	return (int)s.size();
}


int countfields(const string & s, const string & fd)
{
	int cnt = 1;
	int i;
	int sz = (int)fd.size();
	for (i = 0; i<(int)s.size(); ++i)
	{
		if ( strcmpi( s.substr( i, sz ).c_str(), fd.c_str() ) == 0 ) { cnt++; }
	}
	return cnt;
}



string nthfield(const string & s, const string & fd, int fn)
{
	int cnt = 1;
	int po = 0;
	int i;
	int sz = (int)fd.size();
	for (i = 0; i<(int)s.size(); ++i)
	{
		if (strcmpi( s.substr( i, sz ).c_str(), fd.c_str() ) == 0 )
		{
			if (cnt == fn)
			{
				return( s.substr( po, i-po ) );
			} else {
				cnt++;
				po = i+sz;
			}
		}
	}
	if (cnt == fn)
	{
		return( s.substr( po, i-po ) );
	} else {
		return "";
	}
}


bool sml_memcompare(char * m1, char * m2, int ml, bool caseInsensitive)
{
	int i;
	int c1, c2;
	for (i = 0; i<ml; ++i)
	{
		c1 = m1[i];
		c2 = m2[i];
		if (caseInsensitive & c1>64 & c2<91) { c1+=32; }
		if (caseInsensitive & c2>64 & c2<91) { c2+=32; }
		if (c1 != c2) { return false; }
	}
	return true;
}


bool cse(const string & s, const string & ss)
{
	return s == ss;
}

bool cie(const string & s, const string & ss)
{
	return ( strcmpi( s.c_str(), ss.c_str() ) == 0 );
}

string rtrim( const string &s )
{
    string n = s;
    return n.erase( n.find_last_not_of( " " ) + 1 );
}

string ltrim( const string &s )
{
    string n = s;
    return n.erase( 0, s.find_first_not_of( " " ) );
}

string trim( const string &s )
{
    string n = s;
    return ltrim( rtrim( n ) );
}


string reversestring(const string & s)
{
	int i;
	string r;
	for( i = (int)s.size(); i >= 0; --i )
	{
		r += s[i];
	}
	
	return r;
}

/*
char * stringtohex(char * c)
{
	char * alph = new char[16];
	int i,j;
	int sz = strlen(c);
	char * t = new char[(sz*2)+1];
	for (i = 0; i<10; ++i) { alph[i] = 48+i; }
	for (i = 10; i<16; ++i) { alph[i] = 55+i; }
	for (i = 0; i<sz; ++i)
	{
		j = c[i];
		t[(i*2)] = alph[j / 16];
		t[(i*2)+1] = alph[j % 16];
	}
	t[(sz*2)] = 0;
	delete [] alph;
	return t;
}
*/

int strcmpi( const char * s1, const char * s2 )
{
    char c1, c2;
    while ( 1 )
    {
        c1 = tolower( *s1++ );
        c2 = tolower( *s2++ );
        if ( c1 < c2 )
            return -1;
        if ( c1 > c2 )
            return 1;
        if ( c1 == 0 )
            return 0;
    }
}

string lowercase( const string & s )
{
	int i;
	string r ( s );
	for( i = 0; i < ( int ) r.size(); ++i )
	{
		r[ i ] = tolower( r [ i ] );
	}
	return r;
}

string uppercase( const string & s )
{
	int i;
	string r ( s );
	for( i = 0; i < ( int ) r.size(); ++i )
	{
		r[ i ] = toupper( r [ i ] );
	}
	return r;
}

int strcomp( const string & s1, const string & s2 )
{
	return strcmp( s1.c_str(), s2.c_str() );
}

string thatmanystrings( int count, const string & s1 )
{
	string r;
	for( ; count > 0; --count )
	{
		r += s1;
	}
	return r;
}

string replaceall( const string & source, const string & oldstr, const string & newstr )
{
	string r;
	int i,l;
	l = ( int ) oldstr.size();
	for( i = 0; i < ( int ) source.size(); ++i )
	{
		if( strcmpi( source.substr( i, l ).c_str(), oldstr.c_str() ) == 0 )
		{
			r += newstr;
			i += l-1;
		}
		else
		{
			r += source[ i ];
		}
	}
	return r;
}

string replace( const string & source, const string & oldstr, const string & newstr )
{
	string r;
	int i,l;
	l = ( int ) oldstr.size();
	for( i = 0; i < ( int ) source.size(); ++i )
	{
		if( strcmpi( source.substr( i, l ).c_str(), oldstr.c_str() ) == 0 )
		{
			r += newstr;
			i += l-1;
			break;
		}
		else
		{
			r += source[ i ];
		}
	}
	return r + source.substr( i+1, ( int ) source.size() );
}

int countstuff( const string & s1, const string & s2 )
{
	return ( int ) s1.size() - ( ( int ) replaceall( s1, s2, "" ).size() );
}

double val( const string & stringValue )
{
    char **a = NULL;
	return strtod( stringValue.c_str(), a );
}

string str( long double d )
{
    char a[ 255 ] = {0};
	snprintf( a, 254, "%g", d );
	return string(a);
}

string titlecase( const string & s )
{
	char bounds[ 5 ] = { 13, 10, ' ', 34, 39 };
	int i,j;
	bool b;
	string r;
	
	r = toupper( s [ 0 ] );
	
	for( i = 1; i < ( int ) s.size(); ++i )
	{
		b = false;
		for( j = 0; j < 6; ++j )
		{
			if( s [ i - 1 ] == bounds [ j ] )
			{
				b = true;
			}
		}
		if( b )
		{
			r += toupper( s [ i ] );
		}
		else
		{
			r += tolower( s [ i ] );
		}
	}
	return r;
}

string captitle( const string & s )
{
	return titlecase( s );
}

string capallupper( const string & s )
{
	return uppercase( s );
}

string capalllower( const string & s )
{
	return lowercase( s );
}

string capalternating( const string & s )
{
	bool b ( true );
	string r;
	int i;
	
	for( i = 0; i < ( int ) s.size(); ++i ){
		if( b = !b )
		{
			r += toupper( s[ i ] );
		}
		else
		{
			r += tolower( s[ i ] );
		}
	}
	return r;
}

string capinvert( const string & s )
{
	string r;
	int i;
	for( i = 0; i < ( int ) s.size(); ++i )
	{
		if( isupper( s[ i ] ) )
		{
			r += tolower( s[ i ] );
		}
		else
		{
			r += toupper( s[ i ] );
		}
	}
	return r;
}

string capsentence( const string & s )
{
	// It's not ideal, but it works.
	int i;
	bool b (false);
	string r;
	
	r = toupper( s [ 0 ] );
	
	for( i = 1; i < ( int ) s.size(); ++i )
	{
		if( s[ i ] == ' ' || s[ i ] == 13 || s[ i ] == 9 )
		{
			r += s[ i ];
			continue;
		}
		if( s[ i ] == '.' )
		{
			b = true;
			r += '.';
			continue;
		}
		if( b )
		{
			r += toupper( s [ i ] );
			b = false;
		}
		else
		{
			r += s [ i ];
		}
	}
	return r;
}

bool is_in_vector( const string & s, vector < string > & v )
{
	vector < string > :: iterator i;
	for( i = v.begin(); i != v.end(); i ++ )
	{
		if( ( strcomp ( ( * i ), s ) == 0 ) || ( strcomp ( ( * i ) + "s", s ) == 0 ) )
		{
			return true;
		}
	}
	return false;
}

int convert_ordinal(const string & s)
{
	string t = trim(s);
	
	/* special cases */
	if ( cie(t,"last") ) { return -1; }
	if ( cie(t,"any") ) { return -2147483647; }
	if ( cie(t,"some") ) { return -2147483647; }
	if ( cie(t,"mid") ) { return 2147483647; }
	if ( cie(t,"middle") ) { return 2147483647; }
	if ( cie(t,"mdd") ) { return 0x80000000; } /* mdd stands for "m[i]d [rounding] d[own]" */
	
	/* rule out anything with more than one underscore */
	if ( countfields(t, "_") > 2 ) { return -2; }
	
	/* set up table of ordinals */
	int i, m;
	string ss;
	string x = "zeroth first second third fourth fifth sixth seventh eighth ninth tenth eleventh twelfth thirteenth fourteenth fifteenth sixteenth seventeenth eighteenth nineteenth";
	string y = "zeroth tenth twentieth thirtieth fortieth fiftieth sixtieth seventieth eightieth ninetieth";
	string z = "zero ten twenty thirty forty fifty sixty seventy eighty ninety";
	
	/* for ordinals without underscores */
	if ( instr(t, "_") <= 0 )
	{
		/* ordinals of the form ###{st|nd|rd|th} */
		ss = right(t, 2);
		if ( (ss==(string)"st") || (ss==(string)"nd") || (ss==(string)"rd") 
			 || (ss==(string)"th") )
		{
			ss = left(t, t.size()-2);
			for (i=0; i<(signed int)ss.size(); i++)
			{
				if ( (ss[i] < 48) || (ss[i] > 57) ) { continue; }
				else if ( i == (signed int)ss.size()-1 ) { return (int)val(ss); }
			}
		}
		
		if ( instr(x+y, t) <= 0 ) { return -2; }
		
		/* singles (zeroth, first, second, ... nineteenth) */
		for (i = 1; i < 20; i++)
		{
			if ( cie(t,nthfield(x, " ", i)) ) { return i-1; }
		}
		
		/* tens (tenth, twentieth, thirtieth, ... ninetieth) */
		for (i = 1; i < 10; i++)
		{
			if ( cie(t,nthfield(y, " ", i)) ) { return ((i-1)*10); }
		}
		
		return -2;
	}
	
	/* for ordinals with underscores */
	string a = nthfield(t, "_", 1);
	string b = nthfield(t, "_", 2);
	if ( instr(x, b) <= 0 ) { return -2; }
	if ( instr(z, a) <= 0 ) { return -2; }
	for (i = 1; i < 20; i++)
	{
		if ( cie(b,nthfield(x, " ", i)) ) { m = i-1; }
	}
	if ( m < 0 ) { return -2; }
	for (i = 1; i < 10; i++)
	{
		if ( cie(a,nthfield(z, " ", i)) ) { m += ((i-1)*10); }
	}
	if ( m < 20 ) { return -2; }
	return m;
}

char toupper(char c)
{
	return ( ( c > 64 ) && ( c < 91 ) ) ? c + 32 : c;
}

char tolower( char c )
{
	return ( ( c > 96 ) && ( c < 123 ) ) ? c - 32 : c;
}

bool isupper( char c )
{
	return ( ( c > 64 ) && ( c < 91 ) );
}